package edu.gwu.simplenetsim;

import edu.gwu.simplenetsim.*;


/**
 * Interface <code>TransportLayer</code> must be implemented by
 * any class that wants to implement a transport layer.
 *
 * @author Rahul Simha
 * @see StackLayer
 */

public interface TransportLayer extends StackLayer {


    /**
     * Method <code>init</code> takes in a node-ID, and references
     * to the layers above and below.
     *
     * @param nodeNum an <code>int</code> value
     * @param appLayer an <code>ApplicationLayer</code> value
     * @param networkLayer a <code>NetworkLayer</code> value
     */

    public void init (int nodeNum, ApplicationLayer appLayer, NetworkLayer networkLayer);


    /**
     * The application layer calls this to initiate a connection.
     * This is where you handle connection set up.
     *
     * @param src an <code>int</code> value, the source.
     * @param dest an <code>int</code> value, the destination.
     * @param portNum an <code>int</code> value, the port number.
     * @return an <code>int</code> value
     */

    public int openConnection (int src, int dest, int portNum);


    /**
     * Because a connection set up can take time (a few rounds of
     * packets going back and forth), the application layer repeatedly
     * calls <code>isReady</code> to see whether the connection has
     * been set up.
     *
     * @param connID an <code>int</code> value
     * @return a <code>boolean</code> value
     */

    public boolean isReady (int connID);


    /**
     * After a connection is set up, the application sends packets
     * by calling this method. This layer should make a network packet
     * and call the layer below.
     *
     * @param packet a <code>TransportPacket</code> value
     */

    public void sendPacket (TransportPacket packet);


    /**
     * The network layer calls this method when a packet has arrived.
     *
     * @param packet a <code>TransportPacket</code> value
     */

    public void receivePacket (TransportPacket packet);


    /**
     * The application layer is supposed to call <code>closeConnection</code>
     * to terminate the connection. However, we are not doing this now.
     *
     * @param connID an <code>int</code> value
     */

    public void closeConnection (int connID);

}
