// SimpleNetSim.java
//
// Author: Rahul Simha
// July, 2009
//
// A simple network simulator to enable
// writing of transport and network layers

package edu.gwu.simplenetsim;

import edu.gwu.simplenetsim.*;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.util.*;
import java.text.*;
import java.io.*;




/**
 * Class <code>SimpleNetSim</code> is the "main" simulator class,
 * for which reason it is a bit large. This has both the GUI and
 * the guts of the simulator - the data structures, main loop etc.
 * Note also that this class surrounds the basic three (transport,
 * network, datalink) layers from above and below, which is why
 * this class implements <code>PhysicalLayer</code> and 
 * <code>ApplicationLayer</code>. 
 * There are a couple of static methods of use, which we'll document.
 *
 * @author Rahul Simha
 * @see PhysicalLayer
 * @see ApplicationLayer
 * @see MouseListener
 */

public class SimpleNetSim extends JPanel 
    implements PhysicalLayer, ApplicationLayer, MouseListener {

    /**
     *   The port number used for webservers.
     */
    public static final int WEBSERVER_PORT_NUM = 80;

    /**
     *   The port number used for webbrowsers.
     */
    public static final int WEBBROWSER_PORT_NUM = 10000;


    /**
     * How often we update link status, e.g., every 10 time units.
     */
    public static int LINK_UPDATE_INTERVAL = 10;


    ///////////////////////////////////////////////////////////////////////
    // Variables

    // For painting the screen.
    double minX=0, maxX=10, minY=0, maxY=10;      // Bounds.
    int inset=60;                                 // Border.
    String msg = "";                              // Optional msg. 
    double msgX, msgY;

    // A status display at the NORTH location.
    JLabel statusLabel = new JLabel ("    ");

    // Various parameters in the GUI.
    JTextField fileField;                         // Param file. 
    JTextField lossField;                         // Prob[We drop a packet].
    JTextField timeField;                         // Time limit.
    JLabel timeLabel = new JLabel ("0");          // Current time.
    JComboBox animBox;                            // Animation speed.
    JToggleButton debugB;                         // Debug or not?

    // Animation related stuff. Three speeds.
    String[] animStrings = {"None","Slow","Fast"};
    String animType = "None";
    int animSlowSleep = 100;
    int animFastSleep = 25;
    double moveDist = 0.05;
    ArrayList<Edge> animEdgeList;                 // List of "active" edges

    // Some flags to tell us the status of the GUI.
    boolean parametersLoaded = false;
    boolean isPaused = true;
    boolean animate = true;
    boolean resetDone = false;

    // The popup menu when a node is right-clicked.
    JPopupMenu nodePopup;
    Node browserNode;                 // At which node did the click occur?

    // We will drop packets according to lossProb (entered in GUI).
    double lossProb = 0;

    // Max time limit for simulation. Note: this only does one single
    // run and is not appropriate for statistical simulations.
    int timeLimit=10;

    // Need to keep a reference to the main simulator thread in case
    // someone presses "go" twice.
    Thread goThread;

    // We need to make this static because we're going to implement
    // timers using static methods.
    static int time;

    // Network related info.
    int numNodes, numEdges;                    // # nodes, # edges.
    int[][] adjMatrix;                         // Connectivity matrix.
    ArrayList<Node> nodes;                     // Node info. See Node.java
    Edge[][] edges;                            // Edge info. See Edge.java
    ArrayList<Edge> edgeList;                  // We'll also use this.
    ArrayList<Connection> connections;         // All the active connections.
    int numConnections;

    // Packet tracking. Every packet will have a universal ID (for debugging).
    // We will also see how many got delivered.
    int universalPacketID = 0;
    HashSet<String> undeliveredPackets;
    int numTransmitted, numReceived;

    // Since the student writes the layers, we'll upload their code
    // into "Class" objects, from which we'll create instances.
    // Every node will have one instance of each of these classes.
    Class datalinkClass, networkClass, transportClass;
    Class webbrowserClass, webserverClass;

    // These are the filenames, which we will read from the parameter file.
    String datalinkFileName, networkFileName, transportFileName;
    String webbrowserFileName, webserverFileName;

    // We will allow the layers to create timers.
    static PriorityQueue<TimerEvent> timers = new PriorityQueue<TimerEvent>();


    ///////////////////////////////////////////////////////////////////////
    // Constructor - a simple one.

    public SimpleNetSim ()
    {
        // We listen for mouse clicks.
	this.addMouseListener (this);
    }


    ///////////////////////////////////////////////////////////////////////
    // Methods related to reading in the parameters, when the "load"
    // button is clicked. Apologies for the ugly parsing code.

    void load ()
    {
	try {
            // Read line by line.
	    LineNumberReader lnr = new LineNumberReader (new FileReader (fileField.getText()));

	    // Read in path and class files for the layers.
	    readClassFiles (lnr);

	    // Skip to numNodes line.
	    String line = skipToData (lnr);
	    Scanner scanner = new Scanner (line);
	    numNodes = scanner.nextInt ();

	    // Get data for the nodes, one by one (one per line).
	    System.out.println ("numNodes=" + numNodes);
	    nodes = new ArrayList<Node>();
	    line = skipToData (lnr);
	    int i = 0;
	    while (i < numNodes) {
		scanner = new Scanner (line);
		Node node = new Node ();
		node.nodeNum = scanner.nextInt ();
		node.x = scanner.nextDouble ();     // x,y location on screen.
		node.y = scanner.nextDouble ();     // We use 10 x 10 coords.
		i ++;
		nodes.add (node);
		System.out.println ("Node: " + node.nodeNum + " x=" + node.x + " y=" + node.y);
		line = lnr.readLine ();
	    }

	    // Make adjacency matrix.
	    adjMatrix = new int [numNodes][numNodes];
	    edges = new Edge [numNodes][numNodes];

	    // Next, edges.
	    line = skipToData (lnr);
	    scanner = new Scanner (line);
	    numEdges = scanner.nextInt ();
	    System.out.println ("numEdges=" + numEdges);
	    edgeList = new ArrayList<Edge> ();
	    line = skipToData (lnr);
	    i = 0;
	    while (i < numEdges) {
		scanner = new Scanner (line);
		int start = scanner.nextInt ();
		int end = scanner.nextInt ();
		adjMatrix[start][end] = 1;
		adjMatrix[end][start] = 1;
		Edge edge = new Edge ();
		edge.fromNode = start;
		edge.toNode = end;
		edges[start][end] = edge;
		edgeList.add (edge);
		edge = new Edge ();
		edge.fromNode = end;
		edge.toNode = start;
		edges[end][start] = edge;
		edgeList.add (edge);
		Node fromNode = nodes.get (edge.fromNode);
		Node toNode = nodes.get (edge.toNode);
		fromNode.neighbors.add (edge.toNode);
		toNode.neighbors.add (edge.fromNode);
		i ++;
		System.out.println ("Edge: " + start + " " + end);
		line = lnr.readLine ();
	    }

	    // Next, connection info.
	    line = skipToData (lnr);
	    scanner = new Scanner (line);
	    numConnections = scanner.nextInt ();
	    System.out.println ("NumConn=" + numConnections);
	    connections = new ArrayList<Connection>();
	    line = skipToData (lnr);
	    i = 0;
	    while (i < numConnections) {
		scanner = new Scanner (line);
		Connection conn = new Connection ();
		conn.connNum = scanner.nextInt ();
		conn.src = scanner.nextInt ();
		conn.dest = scanner.nextInt ();
		conn.startTime = scanner.nextInt ();
		conn.endTime = scanner.nextInt ();
		conn.packetRate = scanner.nextDouble ();
		connections.add (conn);
		System.out.println ("  " + conn.connNum + " s=" + conn.src + " d=" + conn.dest + " st=" + conn.startTime + " e=" + conn.endTime + " r=" + conn.packetRate);
		i ++;
		line = lnr.readLine ();
	    }
	    
	    status ("Parameters loaded from file " + fileField.getText());
	    parametersLoaded = true;
	    this.repaint ();

	}
	catch (Exception e) {
	    error ("Error reading file");
	    parametersLoaded = false;
	}
    }


    // Skip blank lines or lines beginning with "#".

    String skipToData (LineNumberReader lnr) 
	throws IOException 
    {
	String line = lnr.readLine ();
	while (line != null) {
	    //System.out.println ("skip(): line=[" + line + "]");
	    line = line.trim ();
	    if ( (line.length() == 0) || (line.startsWith ("#")) ) {
		line = lnr.readLine ();
	    }
	    else {
		break;
	    }
	}
	if (line == null) {
	    throw new IOException ();
	}
	return line;
    }


    // Read the class files for the various layers.

    void readClassFiles (LineNumberReader lnr)
	throws IOException 
    {
	try {
	    String line = skipToData (lnr);
	    String fullDirPath = extractFileName (line);
	    System.out.println ("Reading class files from: " + fullDirPath);

            // Data link layer.
	    line = skipToData (lnr);
	    datalinkFileName = extractFileName (line);
	    FileClassLoader fcl = new FileClassLoader (fullDirPath);
	    datalinkClass = Class.forName (datalinkFileName, true, fcl);
	    
            // Network layer.
	    line = lnr.readLine ();
	    networkFileName = extractFileName (line);
	    fcl = new FileClassLoader (fullDirPath);
	    networkClass = Class.forName (networkFileName, true, fcl);
	    
            // Transport layer.
	    line = lnr.readLine ();
	    transportFileName = extractFileName (line);
	    fcl = new FileClassLoader (fullDirPath);
	    transportClass = Class.forName (transportFileName, true, fcl);

            // Web browser.
	    line = lnr.readLine ();
	    webbrowserFileName = extractFileName (line);
	    fcl = new FileClassLoader (fullDirPath);
	    webbrowserClass = Class.forName (webbrowserFileName, true, fcl);

            // Web server.
	    line = lnr.readLine ();
	    webserverFileName = extractFileName (line);
	    fcl = new FileClassLoader (fullDirPath);
	    webserverClass = Class.forName (webserverFileName, true, fcl);
	    
	    System.out.println ("Read class files for layers: " + datalinkFileName + "," + networkFileName + "," + transportFileName + "," + webbrowserFileName + "," + webserverFileName);
	}
	catch (Exception e) {
	    System.out.println (e);
	}
    }


    String extractFileName (String propStr)
    {
	int k = propStr.indexOf ('=');
	if ( (k < 1) || (k >= propStr.length()-1) ) {
	    error ("Error in filename specified: " + propStr);
	    return null;
	}
	String fn = propStr.substring (k+1);
	return fn;
    }


    // This and related methods below create instances of the
    // user classes.

    TransportLayer getTransportLayerInstance ()
    {
	try {
	    return (TransportLayer) transportClass.newInstance();
	}
	catch (Exception e) {
	    System.out.println (e);
	    error ("Could not instantiate TransportLayer class: " + transportFileName);
	    return null;
	}
    }

    NetworkLayer getNetworkLayerInstance ()
    {
	try {
	    return (NetworkLayer) networkClass.newInstance();
	}
	catch (Exception e) {
	    System.out.println (e);
	    error ("Could not instantiate NetworkLayer class: " + networkFileName);
	    return null;
	}
    }

    DatalinkLayer getDatalinkLayerInstance ()
    {
	try {
	    return (DatalinkLayer) datalinkClass.newInstance();
	}
	catch (Exception e) {
	    System.out.println (e);
	    error ("Could not instantiate DatalinkLayer class: " + datalinkFileName);
	    return null;
	}
    }

    WebBrowser getWebBrowserInstance ()
    {
	try {
	    return (WebBrowser) webbrowserClass.newInstance();
	}
	catch (Exception e) {
	    System.out.println (e);
	    error ("Could not instantiate WebBrowser class: " + webbrowserFileName);
	    return null;
	}
    }

    WebServer getWebServerInstance ()
    {
	try {
	    return (WebServer) webserverClass.newInstance();
	}
	catch (Exception e) {
	    System.out.println (e);
	    error ("Could not instantiate WebServer class: " + webserverFileName);
	    return null;
	}
    }



    ///////////////////////////////////////////////////////////////////////
    // Methods related to the core simulation

    void reset ()
    {
	if (! parametersLoaded) {
	    error ("No parameters specified");
	    return;
	}

	// Read time limit.
	try {
	    timeLimit = Integer.parseInt (timeField.getText());
	}
	catch (Exception e) {
	    error ("Error in time limit field");
	    return;
	}

	try {
	    lossProb = Double.parseDouble (lossField.getText());
	}
	catch (Exception e) {
	    error ("Error in loss-prob field");
	    return;
	}

	// Create the layers and initialize.
	for (Node node: nodes) {
	    node.transportLayer = getTransportLayerInstance ();
	    node.networkLayer = getNetworkLayerInstance ();
	    node.datalinkLayer = getDatalinkLayerInstance ();
	    node.webbrowser = getWebBrowserInstance ();
	    node.webserver = getWebServerInstance ();
	    node.transportLayer.init (node.nodeNum, this, node.networkLayer);
	    node.networkLayer.init (node.nodeNum, node.transportLayer, node.datalinkLayer, node.neighbors);
	    node.datalinkLayer.init (node.nodeNum, node.networkLayer, this);
	    node.webbrowser.init (node.nodeNum, node.transportLayer, WEBBROWSER_PORT_NUM, WEBSERVER_PORT_NUM);
	    node.webserver.init (node.nodeNum, node.transportLayer, WEBSERVER_PORT_NUM, WEBBROWSER_PORT_NUM);
	    node.arrivals = new LinkedList<DatalinkPacket> ();
	    Debug.println ("Initialized node: " + node);
	}

	// Set up queues and initialize.
	for (Edge edge: edgeList) {
	    edge.queue = new LinkedList<DatalinkPacket> ();
            edge.linkStatusTotal = 0;
	}

	// Initialize connections.
	for (Connection c: connections) {
	    c.status = Connection.UNOPENED;
	    c.packetCount = 0;
	}

        // Packet tracking.
	universalPacketID = 0;
	undeliveredPackets = new HashSet<String> ();
	numTransmitted = numReceived = 0;

	// Initialize clock.
	time = 0;
	timeLabel.setText ("" + time);

	resetDone = true;
        isPaused = true;
        goThread = null;
	status ("Reset complete: click Go or NextStep");
    }



    void go ()
    {
	if ( (! resetDone) || (goThread !=null) ) {
	    error ("Need to reset before starting or re-starting simulation");
	    return;
	}

	isPaused = false;

        // We make a thread else Swing thread hangs up on the simulation.
        //** Note: this is not a satisfactory solution, just a kludge.
        // A better solution is for each thread to know when it alone is
        // stopped, for which we need thread-specific isPaused variables.
	goThread = new Thread () {
		public void run () {
		    while (time <= timeLimit) {
			doOneStep ();
			if (isPaused) {
			    break;
			}
		    }
		}
	    };

        goThread.start ();

    }


    void pause ()
    {
	isPaused = true;
    }


    void nextStep ()
    {
        // Againg, we thread this to avoid freezing SwingThread.
	Thread thread = new Thread () {
		public void run () {
		    doOneStep ();
		}
	    };
    }


    void doOneStep ()
    {
	if (! resetDone) {
	    error ("Need to reset before starting simulation");
	    return;
	}

        // Increment time.
	time ++;
	timeLabel.setText ("" + time);

	if (time > timeLimit) {
	    status ("Time limit reached: Overall network stats: #transmitted=" + numTransmitted + " numReceived=" + numReceived + " #undelivered=" + undeliveredPackets.size());
	    return;
	}

	Debug.println ("\n\n===================== time=" + time + "===============");

	// Check for timer notifications.
	while (timers.size () > 0) {
	    TimerEvent ev = timers.peek ();
	    if (ev.eventTime <= time) {
		ev = timers.poll ();
		ev.layer.timerBeep (ev.ID, time);
	    }
	    else {
		break;
	    }
	}

	// March through the connections and see if their status changes.
	for (Connection c: connections) {
	    if (time > c.endTime) {
		c.status = Connection.CLOSED;
		Debug.println ("Time expired for " + c);
	    }
	    Node node = nodes.get (c.src);
	    if (time == c.startTime) {
		c.connID = node.transportLayer.openConnection (c.src, c.dest, 0);
		c.status = Connection.WAITING;
		Debug.println ("Changed to WAITING: " + c);
	    }
	    if (c.status == Connection.WAITING) {
		if (node.transportLayer.isReady (c.connID)) {
		    c.status = Connection.ACTIVE;
		    Debug.println ("Changed to ACTIVE: " + c);
		}
	    }
	}

	// For all active connections, generate packet and send
	for (Connection c: connections) {
	    if (c.status == Connection.ACTIVE) {
		if (UniformRandom.uniform() < c.packetRate) {
		    // Packet is generated.
		    universalPacketID ++;
		    String data = "Data: cID=" + c.connID + " univPID=" + universalPacketID;
		    TransportPacket tpacket = new TransportPacket (c.connID, c.src, c.dest, 0, data);
		    undeliveredPackets.add (data);
		    c.packetCount ++;
		    Debug.println ("Packet generated for " + c + " packet=" + tpacket);
		    Node node = nodes.get (c.src);
		    node.transportLayer.sendPacket (tpacket);
		    numTransmitted ++;
		}
	    }
	}

	// For all active webbrowsers, let them do their thing.
	for (Node node: nodes) {
	    node.webbrowser.transmitSTTPRequest ();
	}

	// Check all active webservers.
	for (Node node: nodes) {
	    // If a webserver has data, it'll have a chance to send it.
	    node.webserver.sendData ();
	}


	// Randomize directed edges for variety.
	randomizeEdgeList ();

        // Perform the simple animation.
	if (animate) {
	    animEdgeList = new ArrayList<Edge> ();
	    for (Edge edge: edgeList) {
		if (edge.queue.size() > 0) {
                    edge.currentColor = edge.queue.get(0).color;
		    animEdgeList.add (edge);
		}
	    }
	    animateEdges ();
	}

	// for all edges with packets, place in arrival.
	for (Edge edge: edgeList) {
	    if (edge.queue.size() > 0) {
		DatalinkPacket packet = edge.queue.removeFirst ();
		Node node = nodes.get (edge.toNode);
		Debug.println ("Packet arrival at " + node + " packet=" + packet);
		node.arrivals.add (packet);
	    }
	}

	// Randomize nodes for variety, else node 0 is always first.
	ArrayList<Node> nodeList = randomizeNodes ();

        // Process arrivals.
	for (Node node: nodeList) {
	    while (node.arrivals.size() > 0) {
		DatalinkPacket packet = node.arrivals.removeFirst ();
		Debug.println ("Packet arrival at datalink layer of " + node + " packet=" + packet);
		node.datalinkLayer.receivePacket (packet);
	    }
	}

        // Update link status if need be.
        for (Edge edge: edgeList) {
            if ( (time > 1) && (time % LINK_UPDATE_INTERVAL == 0) ) {
                double avg = edge.linkStatusTotal / LINK_UPDATE_INTERVAL;
                Node node = nodes.get (edge.fromNode);
                node.networkLayer.updateLinkStatus (edge.toNode, new LinkStatus (avg));
                edge.linkStatusTotal = 0;     // Reset.
            }
            else {
                // Accumulate totals.
                edge.linkStatusTotal += edge.queue.size ();
            }
        }
        
    }


    ///////////////////////////////////////////////////////////////////////
    // Animation related stuff.

    void animateEdges ()
    {
	if ( (animEdgeList == null) || (animEdgeList.size() == 0) ) {
	    if (animType.equalsIgnoreCase ("Slow")) {
		this.sleep (8*animSlowSleep);
	    }
	    else {
		this.sleep (8*animFastSleep);
	    }
	    return;
	}
	// First, set the x,y to the start point.
	for (Edge e: animEdgeList) {
	    Node fromNode = nodes.get (e.fromNode);
	    e.animX = fromNode.x;
	    e.animY = fromNode.y;
	    e.animDone = false;
	    Node toNode = nodes.get (e.toNode);
	    double a = Math.atan2 (toNode.y - fromNode.y, toNode.x - fromNode.x);
	    e.animAngle = angleFix (a);
	}

	// Now animate.
	while ( ! animReached() ) {
	    // Move.
	    for (Edge e: animEdgeList) {
		if (! e.animDone) {
		    e.animX += moveDist * Math.cos (e.animAngle);
		    e.animY += moveDist * Math.sin (e.animAngle);
		}
	    }

	    this.repaint ();
	    
	    // Sleep.
	    if (animType.equalsIgnoreCase ("Slow")) {
		this.sleep (animSlowSleep);
	    }
	    else {
		this.sleep (animFastSleep);
	    }
	}
    }

    void sleep (int sleepTime)
    {
	try {
	    Thread.sleep (sleepTime);
	}
	catch (InterruptedException e) {
	    System.out.println (e);
	}
    }

    boolean animReached ()
    {
	boolean reached = true;
	for (Edge e: animEdgeList) {
	    // See if e is done.
	    Node node = nodes.get (e.toNode);
	    if ( (Math.abs (e.animX - node.x) < 0.1) &&
		 (Math.abs (e.animY - node.y) < 0.1) ) {
		// e is done.
		e.animDone = true;
	    }
	    else {
		reached = false;
	    }
	}

	return reached;
    }


    double angleFix (double a)
    {
        // Make each angle an angle between 0 and 2*PI.
        //** Note: this code can be optimized.
        if (a < 0) {
            while (a < 0) {
                a = a + 2*Math.PI;
            }
        }
        else if (a > 2*Math.PI) {
            while (a > 2*Math.PI) {
                a = a - 2*Math.PI;
            }
        }
        return a;
    }


    void randomizeEdgeList ()
    {
	ArrayList<Edge> nextList = new ArrayList<Edge> ();
	// The edgelist size changes, which is why we need the original
	// in the for-loop.
	int size = edgeList.size ();
	for (int i=0; i<size; i++) {
	    int k = UniformRandom.uniform ((int) 0, edgeList.size()-1);
	    Edge edge = edgeList.remove (k);
	    nextList.add (edge);
	}
	if (nextList.size() != size) {
	    // Error.
	    System.out.println ("ERROR: randomizeEdgeList");
	}
	edgeList = nextList;
    }

    ArrayList<Node> randomizeNodes () 
    {
	ArrayList<Node> nodeList = new ArrayList<Node> ();
	for (Node node: nodes) {
	    nodeList.add (node);
	}
	ArrayList<Node> nextList = new ArrayList<Node> ();
	int size = nodeList.size ();
	for (int i=0; i<size; i++) {
	    int k = UniformRandom.uniform ((int) 0, nodeList.size()-1);
	    Node node = nodeList.remove (k);
	    nextList.add (node);
	}
	if (nextList.size() != size) {
	    // Error.
	    System.out.println ("ERROR: randomizeNodeList");
	}
	return nextList;
    }


    ///////////////////////////////////////////////////////////////////////
    // Implements PhysicalLayer interface

    public void transmit (int fromNode, int toNode, DatalinkPacket packet)
    {
	if (adjMatrix[fromNode][toNode] <= 0) {
	    error ("No edge: " + fromNode + "," + toNode);
	    return;
	}
	//** To-Do: Need to check whether packet's info is correct.

	// See if we want to "drop" the packet.
	if (UniformRandom.uniform() < lossProb) {
	    Debug.println ("PACKET LOSS! Dropped packet=" + packet);
	    return;
	}
	// Otherwise, simply add to queue.
	edges[fromNode][toNode].queue.add (packet);
    }


    public void receive (TransportPacket packet)
    {
	if (packet.portNum == WEBSERVER_PORT_NUM) {
	    // Deliver to webserver 
	    Node node = nodes.get (packet.dest);
	    node.webserver.receiveSTTPRequest (packet);
	}
	else if (packet.portNum == WEBBROWSER_PORT_NUM) {
	    // Deliver to webbrowser
	    Node node = nodes.get (packet.dest);
	    node.webbrowser.receiveSTTPResponse (packet);
	}
	else if (! undeliveredPackets.remove (packet.data) ) {
	    error ("Unknown packet: " + packet.data);
	    return;
	}
	numReceived ++;
    }


    ///////////////////////////////////////////////////////////////////////
    // Timers and other static methods.

    public static int setTimer (StackLayer layer, int duration)
    {
	TimerEvent ev  = new TimerEvent (layer, time + duration);
	timers.add (ev);
	return ev.ID;
    }

    public static int getCurrentTime ()
    {
        return time;
    }
    

    ///////////////////////////////////////////////////////////////////////
    // Utility methods.

    void nodeStatus ()
    {
	if (! resetDone) {
	    return;
	}
	// Put something out to status bar regarding browserNode.
	status ("Node " + browserNode.nodeNum + " has " + browserNode.arrivals.size() + " incoming packets in queue");
    }

    void makeBrowser ()
    {
	if (! resetDone) {
	    error ("Need to reset first");
	    return;
	}
	if (! browserNode.webbrowser.isDisplayed() ) {
	    browserNode.webbrowser.display ();
	}
	status ("Web browser for node " + browserNode.nodeNum);
    }


    ///////////////////////////////////////////////////////////////////////
    // GUI construction, paintComponent etc.

    public void status (String msg)
    {
        statusLabel.setForeground (Color.black);
        statusLabel.setText (msg);
    }


    // Report error messages on screen.

    public void error (String str)
    {
        statusLabel.setForeground (Color.red);
        statusLabel.setText ("  " + str);
    }


    public void paintComponent (Graphics g)
    {
        super.paintComponent (g);

        // Background.
        Dimension D = this.getSize ();
	g.setColor (Color.white);
	g.fillRect (0,0, D.width, D.height);
        Graphics2D g2 = (Graphics2D) g;
	g.setColor (Color.black);

	if (nodes == null) {
	    return;
	}

	double xMult = (D.width-2.0*inset) / (maxX-minX);
	double yMult = (D.height-2.0*inset) / (maxY-minY);

        // Message.
        if (msg != null) {
            g.setColor (Color.black);
            int x = (int) ( (msgX-minX) * xMult);
            int y = (int) ((msgY-minY) * yMult);
            g.drawString (msg, inset+x, D.height-y);
        }

	// Now animation.
	//** Change color to packet color.
	g.setColor (Color.green);
	if ( (animEdgeList != null) && (animEdgeList.size() > 0) ) {
	    for (Edge e: animEdgeList) {
		if (! e.animDone) {
                    g.setColor (e.currentColor);
		    // Draw a packet there.
		    int x = (int) ((e.animX-minX) * xMult);
		    int y = D.height - (int) ((e.animY-minY) * yMult);
		    g.fillOval (x-6,y-6,12,12);
		}
	    }
	}

	// First draw edges.
	g.setColor (Color.black);
	for (Edge edge: edgeList) {
	    Node fromNode = nodes.get (edge.fromNode);
	    Node toNode = nodes.get (edge.toNode);
            int x1 = (int) ((fromNode.x-minX) * xMult);
            int y1 = D.height - (int) ((fromNode.y-minY) * yMult);
            int x2 = (int) ((toNode.x-minX) * xMult);
            int y2 = D.height - (int) ((toNode.y-minY) * yMult);
	    g.drawLine (x1,y1, x2,y2);
	}

        // Now nodes.
	g.setColor (Color.red);
	for (Node node: nodes) {
            int x = (int) ((node.x-minX) * xMult);
            int y = D.height - (int) ((node.y-minY) * yMult);
	    node.drawX = x;  
	    node.drawY = y;
	    g.fillOval (x-10,y-10,20,20);
	}

        
    }


    JPanel makeBottomPanel ()
    {
        JPanel panel = new JPanel ();
        
        panel.setLayout (new GridLayout (2,1));
        JPanel sPanel = makeSetupPanel ();
        sPanel.setBorder (BorderFactory.createTitledBorder ("  Set up  "));
        panel.add (sPanel);
        JPanel cPanel = makeControlPanel ();
        cPanel.setBorder (BorderFactory.createTitledBorder ("  Run  "));
        panel.add (cPanel);

        return panel;
    }

    JPanel makeSetupPanel ()
    {
        JPanel panel = new JPanel ();
        
	JLabel fileLabel = new JLabel ("Parameter file:");
	panel.add (fileLabel);
	fileField = new JTextField (10);
	fileField.setText ("test1.dat");
	panel.add (fileField);
	
	JButton loadB = new JButton ("  Load");
	loadB.addActionListener (
	   new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       load ();
		   }
           }
        );
	panel.add (loadB);

	panel.add (new JLabel ("          Packet-loss-prob:"));
	lossField = new JTextField (4);
	lossField.setText (""+lossProb);
	panel.add (lossField);


	panel.add (new JLabel ("           "));
	debugB = new JToggleButton ("Debug");
	debugB.addActionListener (
                 new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       Debug.debug = debugB.isSelected ();
		       if (Debug.debug) {
			   status ("Debugging turned ON");
		       }
		       else {
			   status ("Debugging turned OFF");
		       }
		   }
           }
        );
	panel.add (debugB);

	panel.add (new JLabel ("         Time limit:"));
	timeField = new JTextField (6);
	timeField.setText (""+timeLimit);
	panel.add (timeField);
	
	panel.add (new JLabel ("   Time:"));
	panel.add (timeLabel);

        return panel;
    }

    JPanel makeControlPanel ()
    {
        JPanel panel = new JPanel ();

        panel.add (new JLabel ("Animation: "));
	animBox = new JComboBox (animStrings);
	animBox.addItemListener (
	    new ItemListener () {
		public void itemStateChanged (ItemEvent e)
		{
		    animType = (String) animBox.getSelectedItem ();
		    status ("Animation changed to: " + animType);
		}
	    }
        );
	panel.add (animBox);

        panel.add (new JLabel ("          "));
	JButton resetB = new JButton ("Reset");
	resetB.addActionListener (
	   new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       reset ();
		   }
           }
        );
	panel.add (resetB);

        panel.add (new JLabel ("          "));
	JButton nextB = new JButton ("Next");
	nextB.addActionListener (
	   new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       nextStep ();
		   }
           }
        );
	panel.add (nextB);


        panel.add (new JLabel ("     "));
	JButton goB = new JButton ("Go");
	goB.addActionListener (
	   new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       go ();
		   }
           }
        );
	panel.add (goB);

        panel.add (new JLabel ("     "));
	JButton pauseB = new JButton ("Pause");
	pauseB.addActionListener (
	   new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       pause ();
		   }
           }
        );
	panel.add (pauseB);

        panel.add (new JLabel ("             "));
	JButton quitB = new JButton ("Quit");
	quitB.addActionListener (
	   new ActionListener () {
		   public void actionPerformed (ActionEvent a)
		   {
		       System.exit(0);
		   }
           }
        );
	panel.add (quitB);
        
	return panel;
    }    

    void makeFrame ()
    {
        JFrame frame = new JFrame ();
        frame.setSize (1000, 600);
        frame.setTitle ("Simple Network Simulator");
        Container cPane = frame.getContentPane();
	cPane.add (statusLabel, BorderLayout.NORTH);
        cPane.add (makeBottomPanel(), BorderLayout.SOUTH);
        cPane.add (this, BorderLayout.CENTER);
	makePopup ();
        frame.setVisible (true);
    }

    void makePopup ()
    {
	nodePopup = new JPopupMenu ();
	JMenuItem m = new JMenuItem ("Repaint");
	m.addActionListener (
           new ActionListener () 
	   {
	       public void actionPerformed (ActionEvent a) 
	       {
		   repaint ();
	       }
	   }
	   );
	nodePopup.add (m);

	m = new JMenuItem ("Show node status");
	m.addActionListener (
           new ActionListener () 
	   {
	       public void actionPerformed (ActionEvent a) 
	       {
		   nodeStatus ();
	       }
	   }
	   );
	nodePopup.add (m);

	m = new JMenuItem ("Start browser");
	m.addActionListener (
           new ActionListener () 
	   {
	       public void actionPerformed (ActionEvent a) 
	       {
		   makeBrowser ();
	       }
	   }
	   );
	nodePopup.add (m);
    }


    public void mousePressed (MouseEvent e) 
    {
	// Popup?
	//System.out.println ("MouseP: ev=" + e);
	if (e.isPopupTrigger() || (e.getButton() == 3)) {
	    //System.out.println ("MouseP2");
	    // Find which node, if any.
	    browserNode = null;
	    for (Node node: nodes) {
		double d = distance (e.getX(),e.getY(), node.drawX,node.drawY);
		//System.out.println ("Node: " + node.nodeNum + " d=" + d);
		if (d < 10) {
		    browserNode = node;
		}
	    }
	    if (browserNode != null) {
		nodePopup.show (this, e.getX(), e.getY());
	    }
	}
    }
    
    double distance (double x1, double y1, double x2, double y2)
    {
	return Math.sqrt ((x1-x2)*(x1-x2) + (y1-y2)*(y1-y2));
    }


    // Methods for MouseListener interface.

    public void mouseClicked (MouseEvent e) {}
    public void mouseEntered (MouseEvent e) {}
    public void mouseExited (MouseEvent e) {}
    public void mouseReleased (MouseEvent e) {}
    public void mouseDragged (MouseEvent e) {}
    public void mouseMoved (MouseEvent e) {}



    ///////////////////////////////////////////////////////////////////////
    // main

    public static void main (String[] argv)
    {
	SimpleNetSim net = new SimpleNetSim ();
	net.makeFrame ();
    } 


}



// This implements Comparable so that we can place it in a PriorityQueue.
// The queue is sorted by eventTime of course.

class TimerEvent implements Comparable {

    static int IDCount = 1;
    int eventTime;
    int ID;
    StackLayer layer;

    public TimerEvent (StackLayer layer, int eventTime) 
    {
	this.layer = layer;
	this.eventTime = eventTime;
	ID = IDCount ++;
    }

    public int compareTo (Object obj) 
    {
	if (!equals (obj)) {
	    if (eventTime < ((TimerEvent)obj).eventTime) {
		return -1;
	    }
	    else {
		return 1;
	    }
	}
	else {
	    return 0;
	}
    }

    public boolean equals (Object obj) 
    {
	return (eventTime == ((TimerEvent)obj).eventTime);
    }

}
