// File: DefaultWebBrowser.java
//
// Author: Rahul Simha, 2009.
// This is a rather simple implementation of the STTP protocol. The
// class implements the WebBrowser interface. Because we bring up
// browser as a GUI, it's convenient to have the same class extend
// JPanel and implement WindowListener (to close the window).
// Note: I couldn't get dispose() to work cleanly, not sure why.

import edu.gwu.simplenetsim.*;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;



public class DefaultWebBrowser extends JPanel implements WebBrowser, WindowListener {

    int nodeNum;                       // My node ID, from init().
    int dest;                          // Where the webserver is - we get this from the URL.
    String fileString;                 // The name of the file requested.
    int sourcePortNum;                 // The web browser's port number (10000).
    int destPortNum;                   // The web server's port number (80).
    TransportLayer transportLayer;     // The transport layer to whom I'll send packets.
    int connID;                        // ConnectionID issued by the transport layer.
    boolean sessionActive = false;     // Is this alive? Useful for the GUI.

    // GUI stuff.
    JTextField urlField;               // Where the user types in the URL.
    JTextPane textPane;                // Display for the text.
    JFrame frame;                      // The frame to throw up.
    JLabel endLabel;                   // Possibly use this to indicate download delay? Not used now.
    JLabel statusLabel;                // Feedback to user.
    JButton goButton;                  // Obvious.
    boolean isDisplayed = false;       // Have we put up a frame?


    public void init (int nodeNum, TransportLayer transportLayer, int sourcePortNum, int destPortNum)
    {
        // Get our nodeID, transport layer etc. from the simulator.
	this.nodeNum = nodeNum;
	this.transportLayer = transportLayer;
	this.sourcePortNum = sourcePortNum;
	this.destPortNum = destPortNum;

        // Put the GUI components together, but not the frame (yet).
	textPane = new JTextPane ();
        textPane.setEditable (false);
	textPane.setText ("Welcome to the Default Web Browser");
	urlField = new JTextField (10);
	statusLabel = new JLabel ("DWB: Enter URL and click Get  ");
	endLabel = new JLabel ("  ");
	goButton = new JButton ("Get");
	goButton.addActionListener (
	    new ActionListener () {
		public void actionPerformed (ActionEvent a) 
		{
		    go ();
		}
	    }
        );
	sessionActive = false;
    }


    void go ()
    {
	if (sessionActive) {
            // In case the Go button is clicked multiple times.
	    statusLabel.setText ("Request already sent");
	    return;
	}

	// Read URL from text field, parse and send request.
	String url = urlField.getText();
	if ((url == null) || (url.trim().length() == 0)) {
	    statusLabel.setText ("Empty URL");
	    return;
	}

	url = url.trim ();
	int index = url.indexOf (':');
	if ((index <= 0) || (index == url.length()-1)) {
	    statusLabel.setText ("malformed URL");
	    return;
	}

	String nodeStr = url.substring (0, index);
	fileString = url.substring (index+1);
	//System.out.println ("WebBrowser: nodeStr=[" + nodeStr + "] file=[" + fileString + "]");
	int nodeNum = -1;
	try {
	    nodeNum = Integer.parseInt (nodeStr);
	    dest = nodeNum;
	}
	catch (NumberFormatException e) {
	    System.out.println (e);
	    statusLabel.setText ("malformed URL: couldn't parse node ID");
	    return;
	}

	// Now set up connection.
	connID = transportLayer.openConnection (nodeNum, dest, destPortNum);
	sessionActive = true;

	while (! transportLayer.isReady(connID)) {
	    // Wait.
	    try {
		Thread.sleep (1000);
		Debug.println (" >> sleep");
	    }
	    catch (Exception e) {
		e.printStackTrace ();
	    }
	}

	// Now the connection is ready.

	Debug.println ("DefaultWebBrowser.go(): opened connection ID=" + connID);
	statusLabel.setText ("Opened connection to node " + dest + " for file " + fileString);

	// Next, sent the STTP request.
	transmitSTTPRequest ();
    }


    public void transmitSTTPRequest ()
    {
        // If the transport layer is ready, send the request (a single string).
	if ( (sessionActive) && (transportLayer.isReady (connID)) ) {
	    TransportPacket tpacket = new TransportPacket (connID, nodeNum, dest, destPortNum, fileString);
	    transportLayer.sendPacket (tpacket);
	    Debug.println ("DefaultWebBrowser.transmitSTTPRequest(): sent request packet to transport-layer: transport-packet=" + tpacket);
	    sessionActive = false;
	}
    }

    public void receiveSTTPResponse (TransportPacket responsePacket)
    {
	Debug.println ("DefaultWebBrowser.receiveSTTPResponse(): nodeNum=" + nodeNum + " response-packet=" + responsePacket);
	String str = textPane.getText();
	textPane.setText (str += responsePacket.data + "\n");
    }

    public boolean isDisplayed ()
    {
	return isDisplayed;
    }

    public void display ()
    {
        // Build the GUI.
	frame = new JFrame ();
	frame.setSize (300, 200);
	frame.setTitle ("Simple Web Browser");
        frame.addWindowListener (this);
	Container cPane = frame.getContentPane ();
	JPanel topPanel = new JPanel ();
	topPanel.setLayout (new GridLayout (2,1));
	topPanel.add (statusLabel);
	JPanel subPanel  = new JPanel ();
	subPanel.add (new JLabel ("sttp://"));
	subPanel.add (urlField);
	subPanel.add (goButton);
	subPanel.add (endLabel);
	topPanel.add (subPanel);
	cPane.add (topPanel, BorderLayout.NORTH);
	cPane.add (textPane, BorderLayout.CENTER);
	isDisplayed = true;
	frame.setVisible (true);
    }


    // Window Listener interface.

    public void windowClosing(WindowEvent e) 
    {
	sessionActive = false;
	isDisplayed = false;
	frame.dispose ();
    }

    public void windowClosed(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowActivated(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    

    public static void main (String[] argv)
    {
        //**  WORK IN PROGRESS. I want to create a standalone
        // version of the same, that uses real sockets so that
        // browser and webserver can talk across computers.
	DefaultWebBrowser browser = new DefaultWebBrowser ();
	browser.display ();
    }

}
