/* $Id: MsqlPreparedStatement.java,v 2.2.2.2 1999/01/22 05:33:49 borg Exp $ */
/* Copyright (c) 1998 George Reese */
package com.imaginary.sql.msql;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.FieldPosition;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.Vector;

/**
 * The MsqlPreparedStatement is an mSQL implementation of the 
 * JDBC PreparedStatement interface.  Specifically, it enables an
 * application to execute the same SQL over and over again without
 * repeatedly writing logic to build the SQL statements.  Instead,
 * the application just passes new inputs.  Because mSQL is completely
 * unaware of the concept of a PreparedStatement, the mSQL driver
 * basically hacks it by doing its own parsing and management.
 * There is still a huge advantage to using the PreparedStatement over
 * a regular statement in that you never have to worry about date or
 * String formatting.
 * <BR>
 * Last modified $Date: 1999/01/22 05:33:49 $
 * @version $Revision: 2.2.2.2 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlPreparedStatement extends MsqlStatement
implements PreparedStatement, Runnable {
    private String[]  constants = null;
    private int       parsing   = 0;
    private String    statement = null;
    private String[]  values    = null;
    
    MsqlPreparedStatement(MsqlConnection c, String sql, MsqlInputStream in,
			  MsqlOutputStream out, String enc)
    throws SQLException {
	super(c, in, out, enc);
	statement = sql;
	(new Thread(this)).start();
    }

    public synchronized void clearParameters() throws SQLException {
	while( parsing != 2 ) {
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	for(int i=0; i<values.length; i++) {
	    values[i] = null;
	}
    }

    public synchronized boolean execute() throws SQLException {
	while( parsing != 2 ) {
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	return execute(getSQL());
    }
    
    public synchronized ResultSet executeQuery() throws SQLException {
	while( parsing != 2 ) {
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	return executeQuery(getSQL());
    }

    public synchronized int executeUpdate() throws SQLException {
	while( parsing != 2 ) {
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	return executeUpdate(getSQL());
    }

    private String fixString(String str) {
        if( str.indexOf("'") != -1 ) {
	    StringBuffer buff = new StringBuffer();
            
            for(int i=0; i<str.length(); i++) {
                char c = str.charAt(i);

		switch( c ) {
		case '\'': case '%': case '_':
		    buff.append('\\');
		    break;
		default:
		    break;
                }
		buff.append(c);
            }
	    str = buff.toString();
        }
        return str;
    }

    private String getSQL() throws SQLException {
	String sql = constants[0];

	for(int i=0; i<values.length; i++) {
	    if( values[i] == null ) {
		throw new MsqlException("No value set for parameter " + (i+1) +
				       ".");
	    }
	    sql += values[i];
	    if( (i+1) < constants.length ) {
		sql += constants[i+1];
	    }
	}
	return sql;
    }

    private byte[] getStreamData(InputStream in, int len) throws SQLException {
	byte[] bts = new byte[len];
	byte b;
	
	for(int i=0; i<bts.length; i++) {
	    try {
		b = (byte)in.read();
		if( b == -1 ) {
		    return bts;
		}
	    }
	    catch( IOException e ) {
		throw new MsqlException(e);
	    }
	    bts[i] = b;
	}
	return bts;
    }

    public void run() {
	Vector c = new Vector();
	String sql = statement;
	int ind, count = 0;

	synchronized( this ) {
	    if( parsing != 0 ) {
		return;
	    }
	    parsing = 1;
	}
	while( true ) {
	    ind = sql.indexOf("?");
	    if( ind < 0 ) {
		c.addElement(sql);
		break;
	    }
	    count++;
	    if( ind == 0 ) {
		c.addElement("");
	    }
	    else {
		c.addElement(sql.substring(0, ind));
	    }
	    if( ind < (sql.length() -1) ) {
		sql = sql.substring(ind+1);
	    }
	    else {
		sql = "";
	    }
	}
	values = new String[count];
	constants = new String[c.size()];
	c.copyInto(constants);
	synchronized( this ) {
	    parsing = 2;
	    notifyAll();
	}
    }

    public synchronized void setAsciiStream(int ind, InputStream in, int len)
    throws SQLException {
	byte[] bts;

	validateIndex(ind);
	bts = getStreamData(in, len);
	try {
	    values[ind-1] = new String(bts, "8859_1");
	}
	catch( UnsupportedEncodingException e ) {
	    throw new MsqlException("No ASCII!!!");
	}
    }

    public synchronized void setBigDecimal(int ind, BigDecimal bd)
    throws SQLException {
	validateIndex(ind);
	values[ind-1] = bd.toString();
    }
    
    public synchronized void setBinaryStream(int ind, InputStream in, int len)
    throws SQLException {
	throw new MsqlException("mSQL has no binary data types.");
    }

    public synchronized void setBoolean(int ind, boolean b)
    throws SQLException {
	validateIndex(ind);
	if( b ) {
	    values[ind-1] = "1";
	}
	else {
	    values[ind-1] = "0";
	}
    }

    public synchronized void setByte(int ind, byte b) throws SQLException {
	validateIndex(ind);
	values[ind-1] = ("" + b);
    }

    public synchronized void setBytes(int ind, byte[] b) throws SQLException {
	validateIndex(ind);
	try {
	    values[ind-1] = new String(b, "UTF8");
	}
	catch( UnsupportedEncodingException e ) {
	    throw new MsqlException("UTF8 not supported.");
	}
    }

    public synchronized void setDate(int ind, Date d) throws SQLException {
	SimpleDateFormat fmt = new SimpleDateFormat("dd-MMM-yyyy", Locale.US);
	
	validateIndex(ind);
	values[ind-1] = fmt.format(d);
    }
    
    public synchronized void setDouble(int ind, double d) throws SQLException {
	validateIndex(ind);
	values[ind-1] = ("" + d);
    }
    
    public synchronized void setFloat(int ind, float f) throws SQLException {
	validateIndex(ind);
	values[ind-1] = ("" + f);
    }
    
    public synchronized void setInt(int ind, int x) throws SQLException {
	validateIndex(ind);
	values[ind-1] = ("" + x);
    }

    public synchronized void setLong(int ind, long l) throws SQLException {
	validateIndex(ind);
	values[ind-1] = ("" + l);
    }
    
    public synchronized void setNull(int ind, int type) throws SQLException {
	validateIndex(ind);
	values[ind-1] = "NULL";
    }

    public synchronized void setObject(int ind, Object ob)
    throws SQLException {
	validateIndex(ind);
	values[ind-1] = ob.toString();
	
    }
    
    public synchronized void setObject(int ind, Object ob, int type)
    throws SQLException {
	switch( type ) {
	case Types.CHAR: case Types.VARCHAR: case Types.LONGVARCHAR:
	    if( ob instanceof Date ) {
		setDate(ind, (Date)ob);
	    }
	    else {
		setString(ind, ob.toString());
	    }
	    break;
	default:
	    setObject(ind, ob);
	    break;
	}
    }

    public synchronized void setObject(int ind, Object ob, int type, int scale)
    throws SQLException {
	setObject(ind, ob, type);
    }

    public synchronized void setShort(int ind, short s) throws SQLException {
	validateIndex(ind);
	values[ind-1] = ("" + s);
    }

    public synchronized void setString(int ind, String str)
    throws SQLException {
	validateIndex(ind);
	values[ind-1] = "'" + fixString(str) + "'";
    }

    public synchronized void setTime(int ind, Time t) throws SQLException {
	SimpleDateFormat fmt = new SimpleDateFormat("HH:mm:ss", Locale.US);

	validateIndex(ind);
	values[ind-1] = fmt.format(t);
    }

    public synchronized void setTimestamp(int ind, Timestamp t)
    throws SQLException {
	setLong(ind, t.getTime());
    }

    public synchronized void setUnicodeStream(int ind, InputStream in, int len)
    throws SQLException {
	byte[] bts;

	validateIndex(ind);
	bts = getStreamData(in, len);
	try {
	    values[ind-1] = new String(bts, "UTF8");
	}
	catch( UnsupportedEncodingException e ) {
	    throw new MsqlException("UTF8 not supported.");
	}
    }

    private synchronized void validateIndex(int ind) throws SQLException {
	while( parsing != 2 ) {
	    try { wait(); }
	    catch( InterruptedException e ) { }
	}
	if( ind < 1 ) {
	    throw new MsqlException("Cannot address an index less than 1.");
	}
	else if( ind > values.length ) {
	    throw new MsqlException("Attempted to assign a value to " +
				    "parameter " + ind + " when there are " +
				    "only " + values.length + " parameters.");
	}
    }
}
