/* $Id: MsqlDriver.java,v 2.0 1998/10/21 02:38:47 borg Exp $ */
/* Copyright (c) 1997 George Reese */
package com.imaginary.sql.msql;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.DriverPropertyInfo;
import java.sql.SQLException;
import java.util.Properties;

/**
 * The MsqlDriver class implements the JDBC Driver interface from the
 * JDBC specification.  A Driver is specifically concerned with making
 * database connections via new JDBC Connection instances by responding
 * to URL requests.<BR>
 * Last modified $Date: 1998/10/21 02:38:47 $
 * @version $Revision: 2.0 $
 * @author George Reese (borg@imaginary.com)
 */
public class MsqlDriver implements Driver {
    /****************** Static methods and attributes *****************/
    // The static constructor does according to the JDBC specification.
    // Specifically, it creates a new Driver instance and registers it.
    static {
	try {
	    new MsqlDriver();
	}
	catch( SQLException e ) {
	    e.printStackTrace();
	}
    }

    /****************** Instance methods and attributes *****************/
    /**
     * Constructs an MsqlDriver instance.  The JDBC specification requires
     * the driver then to register itself with the DriverManager.
     * @exception java.sql.SQLException an error occurred in registering
     */
    public MsqlDriver() throws SQLException {
	super();
	DriverManager.registerDriver(this);
    }

    /**
     * Gives the major version for this driver as required by the JDBC
     * specification.
     * @see java.sql.Driver#getMajorVersion
     * @return the major version
     */
    public int getMajorVersion() {
	return 1;
    }

    /**
     * Gives the minor version for this driver as required by the JDBC
     * specification.
     * @see java.sql.Driver#getMinorVersion
     * @return the minor version
     */
    public int getMinorVersion() {
	return 0;
    }

    /**
     * The getPropertyInfo method is intended to allow a generic GUI tool
     * to discover what properties it should prompt a human for in order to
     * get enough information to connect to a database.  Note that depending
     * on the values the human has supplied so far, additional values
     * may become necessary, so it may be necessary to iterate though
     * several calls to getPropertyInfo.
     * @param url The URL of the database to connect to.
     * @param info A proposed list of tag/value pairs that will be sent on
     *          connect open.
     * @return An array of DriverPropertyInfo objects describing possible
     *          properties.  This array may be an empty array if no properties
     *          are required.
     * @exception java.sql.SQLException never actually thrown
     */
    public DriverPropertyInfo[] getPropertyInfo(String url, Properties info)
    throws SQLException {
	return new DriverPropertyInfo[0];
    }
				
    /**
     * Returns true if the driver thinks that it can open a connection
     * to the given URL.  In this case, true is returned if and only if
     * the subprotocol is 'msql'.
     * @param url The URL of the database.
     * @return True if this driver can connect to the given URL.
     * @exception java.sql.SQLException never actually is thrown
     */
    public boolean acceptsURL(String url) throws SQLException {
	if( url.length() < 10 ) {
	    return false;
	}
	else {
	    return url.substring(5,9).equals("msql");
	}
    }

    /**
     * Takes a look at the given URL to see if it is meant for this
     * driver.  If not, simply return null.  If it is, then go ahead and
     * connect to the database.  For the mSQL implementation of JDBC, it
     * looks for URL's in the form of <P>
     * <PRE>
     *     jdbc:msql://[host_addr]:[port]/[db_name]
     * </PRE>
     * @see java.sql.Driver#connect
     * @param url the URL for the database in question
     * @param p the properties object
     * @return null if the URL should be ignored, a new Connection
     * implementation if the URL is a valid mSQL URL
     * @exception java.sql.SQLException an error occurred during connection
     * such as a network error or bad URL
     */
    public Connection connect(String url, Properties p) throws SQLException {
	String host, database, orig = url;
	int i, port;
	
	if( url.startsWith("jdbc:") ) {
	    if( url.length() < 6 ) {
		return null;
	    }
	    url = url.substring(5);
	}
	if( !url.startsWith("msql://") ) {
	    return null;
	}
	if( url.length() < 8 ) {
	    return null;
	}
	url = url.substring(7);
	i = url.indexOf(':');
	if( i == -1 ) {
	    port = 1114;
	    i = url.indexOf('/');
	    if( i == -1 ) {
		throw new SQLException("Invalid mSQL URL: " + orig);
	    }
	    if( url.length() < i+1 ) {
		throw new SQLException("Invalid mSQL URL: " + orig);
	    }
	    host = url.substring(0, i);
	    database = url.substring(i+1);
	}
	else {
	    host = url.substring(0, i);
	    if( url.length() < i+1 ) {
		throw new SQLException("Invalid mSQL URL: " + orig);
	    }
	    url = url.substring(i+1);
	    i = url.indexOf('/');
	    if( i == -1 ) {
		throw new SQLException("Invalid mSQL URL: " + orig);
	    }
	    if( url.length() < i+1 ) {
		throw new SQLException("Invalid mSQL URL: " + orig);
	    }
	    try {
		port = Integer.parseInt(url.substring(0, i));
	    }
	    catch( NumberFormatException e ) {
		throw new SQLException("Invalid port number: " +
				       url.substring(0, i));
	    }
	    database = url.substring(i+1);
	}
	return new MsqlConnection(orig, host, port, database, p);
    }
    
    /**
     * Returns information noting the fact that the mSQL database is not
     * SQL-92 and thus cannot support a JDBC compliant implementation.
     */
    public boolean jdbcCompliant() {
	return false;
    }
}
