function [] = manVisWebGUI(Images, coords, dirPath)
%  
% manVisGUI(Images, coords) creates a web page that uses javascript to 
%                           implement a web enabled, interactive
%                           "mouse-over" style
%                           visualization for an image manifold.
% 
% Images:      an image array which is [height,width,numberOfImages] or,
%              for color images [height,width,3,numberOfImages]
%
% coords:      an array of manifold coordinates for the images which 
%              should be [2, numberOfImages]
%
% dirPath:     pathName to the directory in which web files will be put.
%              This should probably be an empty directory.
%
%  Written and updated by Robert Pless and Richard Souvenir, 2005-2011.
%
%  If you find this very useful, you could cite our review paper about
%  image manifold learning:
% @article{plessSouvenir:iml,
%  author = {Robert Pless and Richard Souvenir},
%  title = {A Survey of Manifold Learning for Images},
%  journal = {IPSJ Transactions on Computer Vision and Applications},
%  year = 2009,
%  volume = 1,
%  pages = {83-94}}
%  http://www.jstage.jst.go.jp/article/ipsjtcva/1/0/83/_pdf
%
%  Also, we are keeping a web page of image manifold applications, please 
%   e-mail pless@wustl.edu with pointers to any web visualizations made
%   through this tool or others, that you would like to share.
%
function [] = manVisWebGUI(Images, coords, dirPath)

% First some lightweight error checking:

% get number of frames, for either color or BW images
if length(size(Images)) == 3
    numFrames = size(Images,3);
else
    numFrames = size(Images,4);
end

if max(Images(:))>1
    Images = Images./max(Images(:));
end

if numFrames ~= size(coords,2)
    error('manVisGUI:mismatchImagesAndCoords',...
          'number of images is not the same as number of manifold points');
end

% add / to the end of dirpath if it isn't already there...
if dirPath(end) ~= '/'
    dirPath = [dirpath '/'];
end

%% Make nearest Neighbor matching data structure

scale = max((max(coords(1,:))-min(coords(1,:))), max(coords(2,:))-min(coords(2,:)));
P(1,:) = (coords(1,:) - min(coords(1,:))) / scale;
P(2,:) = (coords(2,:) - min(coords(2,:))) / scale;

% compute the voronoi diagram of the point set:
[TRI] = delaunay(P(1,:),P(2,:));
[Xi Yi] = meshgrid(0:(1/500):1,0:(1/500):1);
k = dsearch(P(1,:),P(2,:),TRI,Xi(:),Yi(:));

% make this into an index image (so that the nearest neighbor lookups 
% on the interactive interface are simply a lookup.
indexImage = reshape(k,501,501);

% add four extra points very far away so that none of the voronoi regions
% around the orinal points extend to infinity.
Q = [P'; -100 0.5; 100 0.5; 0.5 100; 0.5 -100]'
[V,C] = voronoin(Q');

%% Make a bitmap to show manifold points

bMap = zeros(501,501);                  % init blank bitmap.

                                        % find image locations of points
ind = sub2ind(size(bMap), floor(500*P(2,:))+1, floor(500*P(1,:))+1);

bMap(ind) = 1;                          % set those locations to 1.

H = fspecial('gaussian',[7 7],2);       % make the points "bigger" on the image
H2 = (H > 0.04);
bMap2=conv2(double(bMap),double(H2),'same');
bMap2(bMap2>1)= 1;

% write out the manifold points image, and all the original images with
% known names:

imwrite(bMap2,[dirPath '/isoMapPoints.bmp']);
for ix = 1:numFrames
    filename = [ dirPath 'orig' num2str(ix,'%03d') '.jpg' ];
    if (length(size(Images))==3)
        imwrite(Images(:,:,ix),filename);
    else
        imwrite(Images(:,:,:,ix),filename);
    end  
end

%% Now, write the html/javascript to create the webpage
[fid, message]=fopen([dirPath '/page.html'],'w');

% oh dear god, this is horrible code.  Write into our webpage 
% the javascript functions
% for the switching of each picture.  Some 7 year old in Norway can
% probably improve this for me.  Run this code to see the resulting
% html/javascript and enjoy how inefficient it is.

fprintf(fid,'<SCRIPT LANGUAGE="javascript">\n');
for ix = 1:numFrames
    varName = [ 'im' num2str(ix,'%03d') ];
    textLine = [ varName ' = new Image(' num2str(size(Images,1)) ',' ...
            num2str(size(Images,2)) ')\n' ];    
    fprintf(fid,textLine);
    filename = [ 'orig' num2str(ix,'%03d') '.jpg' ];
    textLine = [ varName '.src = "' filename '"\n' ];
    fprintf(fid,textLine);
end
fprintf(fid,'</SCRIPT>\n');

% ok, now the standard html page bits

% draw the image of manifold points 
fprintf(fid,'<a href=isoMapPoints.bmp><IMG NAME="emp" SRC="isoMapPoints.bmp" ISMAP USEMAP="#ep"></a>\n');
fprintf(fid,'<IMG NAME="var" SRC="orig001.jpg">\n');
fprintf(fid,'<SCRIPT LANGUAGE="javascript">\n');
for ix = 1:numFrames
    funName = [ 'flipTo' num2str(ix,'%03d') ];
    newsrc = [ 'im' num2str(ix,'%03d') '.src'];
    textLine = [ 'function ' funName '() { document["var"].src = ' newsrc '; return true; }\n'];
    fprintf(fid,textLine);
end
fprintf(fid,'</SCRIPT>\n');

% on top of that image, define the regions around each point, which will
% cause a particular image to be displayed
fprintf(fid,'<MAP NAME="ep">\n');

for ix = 1:numFrames
    funName = [ 'flipTo' num2str(ix,'%03d') '()' ];
    textline = ['<area SHAPE=POLY COORDS="'] ;
    fprintf(fid,textline);
    cX = V(C{ix},1);
    cY = V(C{ix},2);
    for jx = 1:size(cX,1)
        tX = floor(cX(jx) *500+1);
        tY = floor(cY(jx) *500+1);
        fprintf(fid, [num2str(tX) ', ' num2str(tY) ' ' ]);
    end
    textline = ['" onMouseOver = "' funName '" ' ];
    fprintf(fid,textline);
    textline = ['ALT = "' num2str(ix) '" ' ];
    fprintf(fid,textline);    
    fprintf(fid,'>\n');
end
fprintf(fid,'</MAP>\n');

% Done.
fclose(fid);

display(['web gui complete.  use a browser to open page ' dirPath 'page.html']);
display(['This page is rather minimal, embed this code in another page or add to it.  Enjoy!']);