% function [] = manVisGUI(Images, coords, [plotOptions])
%  
% manVisGUI(Images, coords) creates an interactive "mouse-over" style
%                           visualization for an image manifold.
% 
% Images:      an image array which is [height,width,numberOfImages] or,
%              for color images [height,width,3,numberOfImages]
%
% coords:      an array of manifold coordinates for the images which 
%              should be [2, numberOfImages]
%
% plotOptions: an optional argument with matlab text string of how the manifold
%              coordinates should be displayed, e.g.: 'r+'
%
% fastest way to test if this works, although these images have no manifold
% structure:
%
%  Images = randn(40,40,50);
%  coords = rand(2,50);
%  manVisGUI(Images,coords);
%
%  Notes. 
%    The if the values in the images are greater than 1, they are
%    scaled down to 1.  
%
%    The manifold coordinates are scaled down so the longest axis has a
%    length of 1.
%
%  Written and updated by Robert Pless and Richard Souvenir, 2005-2011.
%
%  If you find this very useful, you could cite our review paper about
%  image manifold learning:
% @article{plessSouvenir:iml,
%  author = {Robert Pless and Richard Souvenir},
%  title = {A Survey of Manifold Learning for Images},
%  journal = {IPSJ Transactions on Computer Vision and Applications},
%  year = 2009,
%  volume = 1,
%  pages = {83-94}}
%  http://www.jstage.jst.go.jp/article/ipsjtcva/1/0/83/_pdf
%
function [] = manVisGUI(Images,coords, varargin)

%% some lightweight error checking:

% get number of frames, for either color or BW images
if length(size(Images)) == 3
    numFrames = size(Images,3);
else
    numFrames = size(Images,4);
end

if max(Images(:))>1
    Images = Images./max(Images(:));
end

if numFrames ~= size(coords,2)
    error('manVisGUI:mismatchImagesAndCoords',...
          'number of images is not the same as number of manifold points');
end

%% Make nearest Neighbor matching data structure

scale = max((max(coords(1,:))-min(coords(1,:))), max(coords(2,:))-min(coords(2,:)));
P(1,:) = (coords(1,:) - min(coords(1,:))) / scale;
P(2,:) = (coords(2,:) - min(coords(2,:))) / scale;

% compute the voronoi diagram of the point set:
[TRI] = delaunay(P(1,:),P(2,:));
[Xi Yi] = meshgrid(0:(1/500):1,0:(1/500):1);
k = dsearch(P(1,:),P(2,:),TRI,Xi(:),Yi(:));

% make this into an index image (so that the nearest neighbor lookups 
% on the interactive interface are simply a lookup.
indexImage = reshape(k,501,501);

%% Now set up figure for GUI

fH = figure(1);
clf;

% make the left side the visualization of the manifold points.
subplot(1,2,1);
pointAxis = gca;
plot(P(2,:),P(1,:),'b.'); 
axis([0 1 0 1]);

% uncomment below line to number the image points:
% text(P(2,:),P(1,:),num2str((1:numFrames)'));

% Draw first image into the right side of the figure.
subplot(1,2,2);
if length(size(Images)) == 3
    im = repmat(Images(:,:,1),[1 1 3]);
else
    im = Images(:,:,:,1);
end
imHandle = image(im);

%% Now set up data structure that is used by callback function
myData.indexImage = indexImage;
myData.images = Images;
myData.pointAxis = pointAxis;
myData.imageHandle = imHandle;

set(fH,'UserData',myData);  % stash the data in the figure

% tell the figure to react when the mouse moves
set(gcf,'WindowButtonMotionFcn','manVisGUICallback');

% make the image to image transition nicer
set(gcf,'doubleBuffer','on');
